#!/bin/bash

DB="yourDb"         # 실제 데이터베이스 이름으로 변경
USER="postgres"     # 실제 PostgreSQL 사용자명으로 변경
SCHEMA="yourSchema" # 스키마 이름
TABLE="yourTable"   # 생성할 테이블 이름

# 대소문자 구분 문제 해결을 위해 소문자로 변환
SCHEMA_LOWER=$(echo "$SCHEMA" | tr '[:upper:]' '[:lower:]')
TABLE_LOWER=$(echo "$TABLE" | tr '[:upper:]' '[:lower:]')

SRID_ORI=5186       # 원본 좌표계
SRID_NEW=0          # 변환 좌표계 (0이면 좌표변환 안함)
CHARSET="UTF-8"     # 문자 인코딩

SHP_DIR="./"        # shp 파일들이 있는 디렉토리 경로
MAX_JOBS=8          # 병렬 작업 수 (시스템 성능에 따라 조정)


# 로그 파일 설정
LOG_DIR="./"  # 로그 디렉토리 (필요시 경로 변경)
mkdir -p "$LOG_DIR"  # 로그 디렉토리가 없으면 생성
LOG_FILE="$LOG_DIR/shp_import_$(date '+%Y%m%d_%H%M%S').log"

# 로그 함수 정의
logger() {
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] $1" | tee -a "$LOG_FILE"
}

# job control 함수 (병렬 처리 제한)
function wait_for_jobs() {
  local current_jobs
  while true; do
    current_jobs=$(jobs -rp | wc -l)
    if [ "$current_jobs" -lt "$MAX_JOBS" ]; then
      break
    fi
    sleep 0.5
  done
}

# append 함수 정의 (COPY 모드 전용)
function import_append() {
  SHP="$1"
  BASENAME=$(basename "$SHP")
  
  # 처리 시작 시간 기록
  FILE_START_TIME=$(date '+%Y-%m-%d %H:%M:%S')
  FILE_START_TIMESTAMP=$(date +%s)
  
  logger "🚀 [시작] $BASENAME 처리 시작: $FILE_START_TIME"
  
  # COPY 모드로 처리 (가장 빠름)
  if [ "$SRID_NEW" -ne 0 ]; then
      OUTPUT=$(shp2pgsql -D -W "$CHARSET" -s $SRID_ORI:$SRID_NEW "$SHP" $SCHEMA_LOWER.$TABLE_LOWER | psql -q -U "$USER" -d "$DB")
  else
      OUTPUT=$(shp2pgsql -D -W "$CHARSET" -s $SRID_ORI "$SHP" $SCHEMA_LOWER.$TABLE_LOWER | psql -q -U "$USER" -d "$DB")
  fi
  
  EXIT_CODE=$?
  
  # 처리 종료 시간 기록
  FILE_END_TIME=$(date '+%Y-%m-%d %H:%M:%S')
  FILE_END_TIMESTAMP=$(date +%s)
  FILE_DURATION=$((FILE_END_TIMESTAMP - FILE_START_TIMESTAMP))
  
  # 파일별 소요시간을 시:분:초 형식으로 변환
  FILE_HOURS=$((FILE_DURATION / 3600))
  FILE_MINUTES=$(((FILE_DURATION % 3600) / 60))
  FILE_SECONDS=$((FILE_DURATION % 60))
  
  # 1보다 작으면 0으로 표시
  if [ $FILE_HOURS -lt 1 ]; then FILE_HOURS=0; fi
  if [ $FILE_MINUTES -lt 1 ]; then FILE_MINUTES=0; fi
  
  if [ $EXIT_CODE -eq 0 ]; then
      logger "✅ [완료] $BASENAME 처리 완료: $FILE_END_TIME (소요: ${FILE_HOURS}시간 ${FILE_MINUTES}분 ${FILE_SECONDS}초)"
  else
      logger "❌ [실패] $BASENAME 처리 실패: $FILE_END_TIME (소요: ${FILE_HOURS}시간 ${FILE_MINUTES}분 ${FILE_SECONDS}초)"
      
      # 오류 내용을 로그에 기록
      logger "⚠️ 오류 내용: $OUTPUT"
      echo "=== $BASENAME 처리 오류 상세 내용 ===" >> "$LOG_FILE"
      echo "$OUTPUT" >> "$LOG_FILE"
      echo "=====================================" >> "$LOG_FILE"
      
      logger "🔄 다음 파일 처리를 계속 진행합니다."
  fi
}



# 시작 시간 기록
START_TIME=$(date '+%Y-%m-%d %H:%M:%S')
START_TIMESTAMP=$(date +%s)


# SHP 디렉토리 및 파일 존재 확인
if [ ! -d "$SHP_DIR" ]; then
    logger "❌ 오류: SHP 디렉토리가 존재하지 않습니다: $SHP_DIR"
    exit 1
fi

SHP_COUNT=$(find "$SHP_DIR" -name "*.shp" | wc -l)
if [ $SHP_COUNT -eq 0 ]; then
    logger "❌ 오류: SHP 파일이 없습니다: $SHP_DIR"
    exit 1
fi
logger "📁 발견된 SHP 파일 수: $SHP_COUNT개"

# 필수 변수 검증
if [ -z "$DB" ]; then
    logger "❌ 오류: DB 변수가 설정되지 않았습니다. 스크립트 상단에서 DB 변수를 설정해주세요."
    exit 1
fi

if [ -z "$TABLE" ]; then
    logger "❌ 오류: TABLE 변수가 설정되지 않았습니다. 스크립트 상단에서 TABLE 변수를 설정해주세요."
    exit 1
fi

logger "📋 설정된 변수:"
logger "   - 데이터베이스: $DB"
logger "   - 사용자: $USER"
logger "   - 스키마: $SCHEMA"
logger "   - 테이블: $TABLE"

# PostgreSQL 연결 테스트
if ! psql -q -U "$USER" -d "$DB" -c "SELECT 1;" >/dev/null 2>&1; then
    logger "❌ 오류: PostgreSQL 연결 실패. 데이터베이스 연결을 확인해주세요."
    exit 1
fi

# 스키마 존재 확인 및 생성
SCHEMA_EXISTS=$(psql -q -U "$USER" -d "$DB" -t -c "SELECT EXISTS (SELECT 1 FROM information_schema.schemata WHERE schema_name = '$SCHEMA_LOWER');" 2>/dev/null | xargs)

if [ "$SCHEMA_EXISTS" != "t" ]; then
    logger "📋 스키마 $SCHEMA_LOWER 생성 중..."
    SCHEMA_OUTPUT=$(psql -q -U "$USER" -d "$DB" -c "CREATE SCHEMA IF NOT EXISTS $SCHEMA_LOWER;")
    if [ $? -ne 0 ]; then
        logger "❌ 스키마 생성 실패: $SCHEMA_OUTPUT"
        exit 1
    fi
fi

# PostGIS 확장 확인
POSTGIS_EXISTS=$(psql -q -U "$USER" -d "$DB" -t -c "SELECT EXISTS (SELECT 1 FROM pg_extension WHERE extname = 'postgis');" 2>/dev/null | xargs)

if [ "$POSTGIS_EXISTS" != "t" ]; then
    logger "⚠️ PostGIS 확장 설치 중..."
    POSTGIS_OUTPUT=$(psql -q -U "$USER" -d "$DB" -c "CREATE EXTENSION IF NOT EXISTS postgis;")
    if [ $? -ne 0 ]; then
        logger "❌ PostGIS 확장 설치 실패: $POSTGIS_OUTPUT"
        logger "⚠️ PostGIS 확장을 수동으로 설치해주세요."
        exit 1
    fi
fi

# 좌표계 설정 확인 및 로깅
if [ "$SRID_NEW" -ne 0 ]; then
    logger "🗺️ 좌표변환 모드: $SRID_ORI → $SRID_NEW"
else
    logger "🗺️ 원본좌표계 사용: $SRID_ORI"
fi

logger "=================================="
logger "🚀 Import 시작: $START_TIME"
logger "=================================="

# 성능 최적화 설정 로깅
logger "⚡ 성능 최적화 설정:"
logger "   - COPY 모드: ✅ 활성화 (INSERT 대비 10-100배 빠름)"
logger "   - 병렬 작업 수: $MAX_JOBS개"

# 테이블 존재 여부 확인
TABLE_EXISTS=$(psql -q -U "$USER" -d "$DB" -t -c "SELECT EXISTS (SELECT 1 FROM information_schema.tables WHERE table_schema = '$SCHEMA_LOWER' AND table_name = '$TABLE_LOWER');" 2>/dev/null | xargs)

# 모든 SHP 파일을 배열로 수집 (공백이 포함된 파일명도 안전하게 처리)
mapfile -t SHP_FILES < <(find "$SHP_DIR" -name "*.shp" | sort)
TOTAL_FILES=${#SHP_FILES[@]}

# 배열이 비어있거나 null인 경우 체크
if [ ${#SHP_FILES[@]} -eq 0 ] || [ -z "${SHP_FILES[0]}" ]; then
    logger "❌ 오류: 유효한 SHP 파일을 찾을 수 없습니다."
    exit 1
fi

if [ "$TABLE_EXISTS" = "t" ]; then
    logger "📋 테이블 $SCHEMA_LOWER.$TABLE_LOWER이 이미 존재합니다. Append 모드로 진행합니다."
    logger "🔄 총 $TOTAL_FILES개 파일을 append 모드로 처리합니다..."
    
    # 모든 파일을 append 모드로 병렬 처리
    for shp in "${SHP_FILES[@]}"; do
        if [ -n "$shp" ]; then
            wait_for_jobs
            import_append "$shp" &
            logger "🔄 백그라운드 작업 시작: $(basename "$shp")"
        fi
    done
    wait
    logger "✅ 모든 append 작업 완료"
else
    logger "📋 테이블 $SCHEMA_LOWER.$TABLE_LOWER이 존재하지 않습니다. 새로 생성합니다."
    logger "🔄 총 $TOTAL_FILES개 파일을 처리합니다..."
    
    # 첫 번째 파일로 테이블 생성
    if [ $TOTAL_FILES -gt 0 ]; then
        FIRST_FILE="${SHP_FILES[0]}"
        FIRST_BASENAME=$(basename "$FIRST_FILE")
        
        # 테이블 생성 시작 시간 기록
        TABLE_START_TIME=$(date '+%Y-%m-%d %H:%M:%S')
        TABLE_START_TIMESTAMP=$(date +%s)
        
        logger "🚀 [시작] 테이블 구조 생성 시작: $FIRST_BASENAME - $TABLE_START_TIME"
        
        # 좌표변환 여부에 따라 shp2pgsql 명령어 결정 (테이블 구조만 생성, 데이터는 나중에)
        if [ "$SRID_NEW" -ne 0 ]; then
            OUTPUT=$(shp2pgsql -p -W "$CHARSET" -s $SRID_ORI:$SRID_NEW "$FIRST_FILE" $SCHEMA_LOWER.$TABLE_LOWER | psql -q -U "$USER" -d "$DB")
        else
            OUTPUT=$(shp2pgsql -p -W "$CHARSET" -s $SRID_ORI "$FIRST_FILE" $SCHEMA_LOWER.$TABLE_LOWER | psql -q -U "$USER" -d "$DB")
        fi
        EXIT_CODE=$?
        
        # 테이블 생성 종료 시간 기록
        TABLE_END_TIME=$(date '+%Y-%m-%d %H:%M:%S')
        TABLE_END_TIMESTAMP=$(date +%s)
        TABLE_DURATION=$((TABLE_END_TIMESTAMP - TABLE_START_TIMESTAMP))
        
        # 테이블 생성 소요시간을 시:분:초 형식으로 변환
        TABLE_HOURS=$((TABLE_DURATION / 3600))
        TABLE_MINUTES=$(((TABLE_DURATION % 3600) / 60))
        TABLE_SECONDS=$((TABLE_DURATION % 60))
        
        # 1보다 작으면 0으로 표시
        if [ $TABLE_HOURS -lt 1 ]; then TABLE_HOURS=0; fi
        if [ $TABLE_MINUTES -lt 1 ]; then TABLE_MINUTES=0; fi

        if [ $EXIT_CODE -eq 0 ]; then
            logger "✅ [완료] 테이블 구조 생성 완료: $TABLE_END_TIME (소요: ${TABLE_HOURS}시간 ${TABLE_MINUTES}분 ${TABLE_SECONDS}초)"
            
            # 모든 파일을 한꺼번에 병렬 처리로 데이터 적재 (첫 번째 파일 포함)
            logger "🔄 모든 $TOTAL_FILES개 파일을 병렬 처리로 데이터 적재합니다..."
            for shp in "${SHP_FILES[@]}"; do
                if [ -n "$shp" ]; then
                    wait_for_jobs
                    import_append "$shp" &
                    logger "🔄 백그라운드 작업 시작: $(basename "$shp")"
                fi
            done
            wait
            logger "✅ 모든 데이터 적재 작업 완료"
        else
            logger "❌ [실패] 테이블 구조 생성 실패: $TABLE_END_TIME (소요: ${TABLE_HOURS}시간 ${TABLE_MINUTES}분 ${TABLE_SECONDS}초)"
            echo "$OUTPUT" >> "$LOG_FILE"
            exit 1
        fi
    else
        logger "❌ 오류: 처리할 SHP 파일이 없습니다."
        exit 1
    fi
fi

# 병렬 처리 완료 확인 및 요약
logger "🔄 병렬 처리 완료 확인 중..."
logger "📊 처리된 파일 수: $TOTAL_FILES개"

# 데이터 입력 완료 시간 기록
DATA_END_TIME=$(date '+%Y-%m-%d %H:%M:%S')
DATA_END_TIMESTAMP=$(date +%s)
DATA_DURATION=$((DATA_END_TIMESTAMP - START_TIMESTAMP))

# 데이터 입력 시간을 시:분:초 형식으로 변환
DATA_HOURS=$((DATA_DURATION / 3600))
DATA_MINUTES=$(((DATA_DURATION % 3600) / 60))
DATA_SECONDS=$((DATA_DURATION % 60))

# 1보다 작으면 0으로 표시
if [ $DATA_HOURS -lt 1 ]; then DATA_HOURS=0; fi
if [ $DATA_MINUTES -lt 1 ]; then DATA_MINUTES=0; fi

# 3. (선택) 인덱스 재생성 (권장: 한 번에 생성)
INDEX_START_TIME=$(date '+%Y-%m-%d %H:%M:%S')
INDEX_START_TIMESTAMP=$(date +%s)

logger "🔍 공간 인덱스 생성 시작: $INDEX_START_TIME"

# 공간 인덱스 생성 (PostGIS 확장이 활성화되어 있어야 함)
INDEX_OUTPUT=$(psql -q -U "$USER" -d "$DB" -c "CREATE INDEX IF NOT EXISTS ${TABLE_LOWER}_geom_idx ON $SCHEMA_LOWER.$TABLE_LOWER USING GIST (geom);")
INDEX_EXIT_CODE=$?

if [ $INDEX_EXIT_CODE -ne 0 ]; then
    # 오류 내용을 로그에 기록
    logger "⚠️ 인덱스 생성 오류: $INDEX_OUTPUT"
    echo "=== 공간 인덱스 생성 오류 상세 내용 ===" >> "$LOG_FILE"
    echo "$INDEX_OUTPUT" >> "$LOG_FILE"
    echo "=====================================" >> "$LOG_FILE"
    
    logger "🔄 인덱스 생성에 실패했지만 스크립트는 계속 진행됩니다."
fi

# 인덱스 재생성 완료 시간 기록
INDEX_END_TIME=$(date '+%Y-%m-%d %H:%M:%S')
INDEX_END_TIMESTAMP=$(date +%s)
INDEX_DURATION=$((INDEX_END_TIMESTAMP - INDEX_START_TIMESTAMP))

# 인덱스 재생성 시간을 시:분:초 형식으로 변환
INDEX_HOURS=$((INDEX_DURATION / 3600))
INDEX_MINUTES=$(((INDEX_DURATION % 3600) / 60))
INDEX_SECONDS=$((INDEX_DURATION % 60))

# 1보다 작으면 0으로 표시
if [ $INDEX_HOURS -lt 1 ]; then INDEX_HOURS=0; fi
if [ $INDEX_MINUTES -lt 1 ]; then INDEX_MINUTES=0; fi

logger "=================================="
logger "✅ 작업 완료"
logger "📋 대상테이블: $SCHEMA_LOWER.$TABLE_LOWER"
if [ "$SRID_NEW" -ne 0 ]; then
    logger "🗺️ 좌표계: $SRID_ORI → $SRID_NEW (변환됨)"
else
    logger "🗺️ 좌표계: $SRID_ORI (원본)"
fi
logger "⏱️ 데이터 입력 소요시간: ${DATA_HOURS}시간 ${DATA_MINUTES}분 ${DATA_SECONDS}초"
logger "⏱️ 공간 인덱스 소요시간: ${INDEX_HOURS}시간 ${INDEX_MINUTES}분 ${INDEX_SECONDS}초"
logger "=================================="